## ASSET accumulation parameters


# Prelims -----------------------------------------------------------------

rm(list=ls())
gc()



# Get dimensions of parameters -------------------------------

## years of projections
year <- c(2018:2050)

## age groups from starting cohorts file
starting_cohorts <- qread("./Input data/starting_cohorts_aiwbh_p.qs")

age_grp <- unique(starting_cohorts$age_grp) 

age_grp_concord <- starting_cohorts %>% 
  distinct(age_grp, age_grp2)


inc_grp <- unique(starting_cohorts$total_inc_qtile)



# Read in parameters from other scripts -----------------------------------

## OTHER ASSETS
##  returns based on assumed rates of return and HILDA asset allocations
other_asset_returns <- qread("./Input data/other_asset_returns_a.qs") %>% 
  select(age_grp, other_asset_return_param = other_assets_return_rate_smooth)

## saving from HES data
other_asset_saving <- qread("./Input data/saving_other_aih.qs") %>% 
  select(age_grp, total_inc_qtile, homeowner, other_asset_saving_param = saving_rate_other )

## drawdowns other
other_asset_drawdown <- qread("./Input data/drawdown_other_ah.qs") %>% 
  select(age_grp2, homeowner, other_asset_drawdown_param = drawdown_rate_smooth)


## HOUSING ASSETS (excluding transition probs)
## value of housing when purchased
new_housing_value <- qread("./Input data/newhousing_value_projection_i.qs") %>% 
  select(year, total_inc_qtile, new_housing_val_param = newhome_price)

## upsizing rate
housing_asset_saving <- qread("./Input data/saving_housing_a.qs") %>% 
  select(age_grp2, housing_asset_saving_param = upsize_rate_housing)

## housing debt to assets ratio
housing_debt_to_asset <- qread("./Input data/newhousing_debt_ratio_ai.qs") %>% 
  select(age_grp2, total_inc_qtile, housing_debt_to_asset_param = housing_debt_to_asset_smooth)

## existing housing debt reduces over time
housing_debt_reduction <- qread("./Input data/debt_reduction_housing_a.qs") %>% 
  select(age_grp2, housing_debt_reduc_param = debt_reduction)




# Rates of return params --------------------------------------------------

## super returns based on RIR assumptions
## housing returns based on 20 year average from Vanguard data and RBA 30 year analysis
## other asset based on "other wealth" asset allocations across ages

return_rates <- expand_grid(year, age_grp) %>%
  mutate(housing_asset_return_param = 0.07, 
         super_asset_return_param = case_when(
           age_grp>="[65,70)" ~ 0.0535, ## retirement phase, net of fees and tax
           age_grp<"[65,70)" ~ 0.06 ## accumulation phase, net of fees and tax
         )
  ) %>% 
  left_join(other_asset_returns)



# Super drawdown rate by year, asset type, age group ------------------------

super_asset_drawdown <- expand_grid(age_grp) %>%
  mutate(super_asset_drawdown_param = case_when(
    ## assume everyone starts withdrawing super when reach 65 age group
    age_grp>="[65,70)" & age_grp<="[70,75)" ~ 0.05, ## minimum drawdowns https://www.ato.gov.au/Rates/Key-superannuation-rates-and-thresholds/?page=8
    age_grp=="[75,80)"                      ~ 0.06, 
    age_grp=="[80,85)"                      ~ 0.07, 
    age_grp=="[85,90)"                      ~ 0.09, 
    age_grp=="[90,95)"                      ~ 0.11, 
    age_grp>="[95,100)"                     ~ 0.14, 
    TRUE ~ 0
  ))



# # Bequest-specific saving rate ----------------------

## assume same as saving from income into other assets in base case

bequest_other_asset_saving <- other_asset_saving %>% 
  rename(bequest_other_asset_saving_param = other_asset_saving_param)


# # Gift-specific saving rate -------------------------

## assume same as saving from income into other assets in base case

gift_other_asset_saving <- other_asset_saving %>% 
  rename(gift_other_asset_saving_param = other_asset_saving_param)



# Put all parameters together ---------------------------------------------

wealth_accum_params <- expand_grid(year, age_grp_concord, total_inc_qtile=as.numeric(inc_grp), homeowner=c(0,1)) %>% 
  left_join(return_rates) %>% 
  left_join(super_asset_drawdown) %>% 
  left_join(other_asset_saving) %>% 
  left_join(other_asset_drawdown) %>% 
  left_join(new_housing_value %>% mutate(total_inc_qtile=as.numeric(total_inc_qtile))) %>% 
  left_join(housing_asset_saving) %>% 
  left_join(housing_debt_to_asset %>% mutate(total_inc_qtile=as.numeric(total_inc_qtile))) %>% 
  left_join(housing_debt_reduction) %>% 
  left_join(bequest_other_asset_saving) %>% 
  left_join(gift_other_asset_saving) %>% 
  mutate(across(c(housing_debt_to_asset_param, housing_asset_saving_param, housing_debt_reduc_param), ~ifelse(is.na(.x), 0, .x)))

#names(wealth_accum_params)

qsave(wealth_accum_params, "./Input data/wealth_accum_params_aiwbh.qs")